clear;
addpath('./EM_sources/');
addpath('./Scatterer_discretization/');
addpath('./CDDA_solution/');
addpath('./Scattering_calculations/');

%% Sphere with chiral shell example (single wavelength)

% This script illustrates how the CDDA functions must be employed in order 
% to do scattering calculations when one or more scatterer are present. 
% The script is clearly separated en three parts. The first one consist 
% in the definition of the diferent parameters that describe the scatterer 
% and the illumination source. The second consist in solving the coupling 
% of the dipolar units. The third is the part where scattering calculations
% are done by means of the solution obtained from the coupled equations.

% The problem treated here is the following: scattering produced by a
% sphere surrounded by a finite shell with circular birrefringence 
% illuminated with a plane wave. The circular birrefringence is quantified 
% by the mixed magnetic-electric susceptibility chi, i.e., the refractive 
% index (RI) for right circularly polarized ligth (CPL) is nR = n0 + chi, 
% while for left CPL is nL = n0 - chi.

% The results obtained here can be directly compared with an analitic
% solution given for example at [1]. More details conernign the DDA method
% are given at [2].


% [1] Bohren, C. F. (1975). Scattering of electromagnetic waves by an 
% optically active spherical shell. The Journal of Chemical Physics, 
% 62(4), 1566-1571.

% [2]Rosales, S. A., Albella, P., González, F., Gutierrez, Y., & Moreno, F. 
% (2021). CDDA: extension and analysis of the discrete dipole approximation 
% for chiral systems. Optics Express, 29(19), 30020-30034.


%%%%%%%%%                                        %%%%%%%%%
%%%%%%%%% Definition of scatterer and plane wave %%%%%%%%%
%%%%%%%%%                                        %%%%%%%%%

Nscat = 2; % number of scatterers
a     =  zeros(1,2);
nsc   = a;
chi   = a;


%%%% Sphere parameters %%%%
r1        = 80;     % Radius  
nsc(1)    = 1.4;    % RI
chi(1)    = 0;
N1        = 200;    % Number of dipoles desired for the discretization. The 
                  % larger this number the smaller is the error, but longer
                  % the calculation
pos0sp  = [0,0,0]; % Position of the center of mass of the sphere

%%%% Shell parameters %%%%
r2a     = 80;     % Inner radius  
r2b     = 90;     % Outer radius
nsc(2)  = 1.4;    % RI for unpolarized light (mean).
chi(2)  = 1e-4;   % nR = nsc + chi, nL = nsc - chi. Due to chirality of the
                  % sphere material
N2      = N1*(r2b^3-r2a^3)/r1^3; % Number of dipoles to mantain the same
                                  % lattice constant (not compulsory)
pos0sh  = [0,0,0]; % Position of the center of mass of the shell
  
nm     = 1.3;     % Surrounding media RI.

%%%% Plane wave parameters %%%%
E      = 1;               % Amplitude
theta  = 0;               % First angle of propagation direction definition
phi    = 0;               % Second angle of propagation direct. def.
                          % If theta = 0, the direction of propagation is
                          % [0,0,1] (cartesian) whatever the value of phi
                          % is.
                          % The plane wave satisfies the null-divergence
                          % property i.e. k.E = 0.
                          
angle  = pi/4;            % Polarization orientation. The case 
                          % theta = 0, angle = 0 is the x polarization. 
phase  = pi/2;            % Phase diference between the two ortogonal
                          % components of the electric field.
lambda = 300;             % Wavelength
k0     = 2*pi*nm/lambda;  % Wavenumber


%%%%%%%%%                                              %%%%%%%%%
%%%%%%%%% Generation and solution of coupled equations %%%%%%%%%
%%%%%%%%%                                              %%%%%%%%%

tic

%%%% Discretization of the scatterers %%%%
[pos1,a(1),Nd(1)]          = spherical_distribution_dipoles(r1, N1,...
                            pos0sp);
[pos2,a(2),Nd(2)]          = spherical_shell_distribution_dipoles(r2a,...
                            r2b, N2, pos0sh);

pos = vertcat(pos1,pos2);
[alphae, alphah, G]   = polarizabilities(nsc.^2, nm^2, 1, 1,...
                            chi, a, k0, Nd);


%%%% Definition of the illumination %%%%
[EH01,vk]  = plane_wave(E,angle, phase,theta,phi);  %% Right CPL
[EH02]     = plane_wave(E,angle,-phase,theta,phi);  %% Left CPL


%%%% Total incident field in each dipole (DDA solution) %%%%
[EH1] = CDDA_solution(pos,alphae,alphah,G,EH01,k0,nm^2);
[EH2] = CDDA_solution(pos,alphae,alphah,G,EH02,k0,nm^2);

%%%%%%%%%                         %%%%%%%%%
%%%%%%%%% Scattering calculations %%%%%%%%%
%%%%%%%%%                         %%%%%%%%%


zc  = 1e6;
rff = zc*vk.'; % Position arbitrarily far from the sphere in the foward
               % direction
               
E1  = far_fields(pos,alphae,alphah,G,EH1,k0,rff,nm^2); % E-far field
E2  = far_fields(pos,alphae,alphah,G,EH2,k0,rff,nm^2); % E-far field

Eff1     = norm(E1)^2;
Eff2     = norm(E2)^2;


rnf = r2b*vk.'; % Position on the surface of the sphere in the foward
              % direction        
E1  = near_fields(pos,alphae,alphah,G,EH1,EH01,k0,rnf,nm^2,1); % E-near field
E2  = near_fields(pos,alphae,alphah,G,EH2,EH02,k0,rnf,nm^2,1); % E-near field

Enf1     = norm(E1)^2;
Enf2     = norm(E2)^2;


%%% Cross sections %%%
[Cextr,Cscar,Cabsr] = cross_sections(pos,alphae,alphah,G,EH1,EH01,k0,nm^2);
[Cextl,Cscal,Cabsl] = cross_sections(pos,alphae,alphah,G,EH2,EH02,k0,nm^2);


toc 


%% Sphere with chiral shell example (multiple wavelengths)

%%%%%%%%%                                        %%%%%%%%%
%%%%%%%%% Definition of scatterer and plane wave %%%%%%%%%
%%%%%%%%%                                        %%%%%%%%%

Nscat = 2; % number of scatterers
a     =  zeros(1,2);
nsc   = a;
chi   = a;


%%%% Sphere parameters %%%%
r1        = 80;     % Radius  
nsc(1)    = 1.4;    % RI
chi(1)    = 0;
N1        = 300;    % Number of dipoles desired for the discretization. The 
                  % larger this number the smaller is the error, but longer
                  % the calculation
pos0sp  = [0,0,0]; % Position of the center of mass of the sphere

%%%% Shell parameters %%%%
r2a     = 80;     % Inner radius  
r2b     = 90;     % Outer radius
nsc(2)  = 1.4;    % RI for unpolarized light (mean).
chi(2)  = 1e-4;   % nR = nsc + chi, nL = nsc - chi. Due to chirality of the
                  % sphere material
N2      = N1*(r2b^3-r2a^3)/r1^3; % Number of dipoles to mantain the same
                                  % lattice constant (not compulsory)
pos0sh  = [0,0,0]; % Position of the center of mass of the shell
  
nm     = 1.3;     % Surrounding media RI.

%%%% Plane wave parameters %%%%
E      = 1;               % Amplitude
theta  = 0;               % First angle of propagation direction definition
phi    = 0;               % Second angle of propagation direct. def.
                          % If theta = 0, the direction of propagation is
                          % [0,0,1] (cartesian) whatever the value of phi
                          % is.
                          % The plane wave satisfies the null-divergence
                          % property i.e. k.E = 0.
                          
angle  = pi/4;            % Polarization orientation. The case 
                          % theta = 0, angle = 0 is the x polarization. 
phase  = pi/2;            % Phase diference between the two ortogonal
                          % components of the electric field.
lambda = 250:450;             % Wavelength
k0     = 2*pi*nm./lambda;  % Wavenumber


%%%%%%%%%                                              %%%%%%%%%
%%%%%%%%% Generation and solution of coupled equations %%%%%%%%%
%%%%%%%%%                                              %%%%%%%%%

tic

%%%% Discretization of the scatterers %%%%
[pos1,a(1),Nd(1)]          = spherical_distribution_dipoles(r1, N1,...
                            pos0sp);
[pos2,a(2),Nd(2)]          = spherical_shell_distribution_dipoles(r2a,...
                            r2b, N2, pos0sh);

pos = vertcat(pos1,pos2);
[alphae, alphah, G]   = polarizabilities(nsc.^2, nm^2, 1, 1,...
                            chi, a, k0, Nd);


%%%% Definition of the illumination %%%%
[EH01,vk]  = plane_wave(E,angle, phase,theta,phi);  %% Right CPL
[EH02]     = plane_wave(E,angle,-phase,theta,phi);  %% Left CPL


%%%% Total incident field in each dipole (DDA solution) %%%%
[EH1] = CDDA_solution(pos,alphae,alphah,G,EH01,k0,nm^2);
[EH2] = CDDA_solution(pos,alphae,alphah,G,EH02,k0,nm^2);

%%%%%%%%%                         %%%%%%%%%
%%%%%%%%% Scattering calculations %%%%%%%%%
%%%%%%%%%                         %%%%%%%%%


zc  = 1e6;
rff = zc*vk.'; % Position arbitrarily far from the sphere in the foward
               % direction
               
E1  = far_fields(pos,alphae,alphah,G,EH1,k0,rff,nm^2); % E-far field
E2  = far_fields(pos,alphae,alphah,G,EH2,k0,rff,nm^2); % E-far field

Eff1     = norm(E1)^2;
Eff2     = norm(E2)^2;


rnf = r2b*vk.'; % Position on the surface of the sphere in the foward
              % direction        
E1  = near_fields(pos,alphae,alphah,G,EH1,EH01,k0,rnf,nm^2,1); % E-near field
E2  = near_fields(pos,alphae,alphah,G,EH2,EH02,k0,rnf,nm^2,1); % E-near field

Enf1     = norm(E1)^2;
Enf2     = norm(E2)^2;


%%% Cross sections %%%
[Cextr,Cscar,Cabsr] = cross_sections(pos,alphae,alphah,G,EH1,EH01,k0,nm^2);
[Cextl,Cscal,Cabsl] = cross_sections(pos,alphae,alphah,G,EH2,EH02,k0,nm^2);


toc 